import {
  Directive,
  ElementRef,
  HostListener,
  Input,
  OnChanges,
  Optional,
  SimpleChanges
} from '@angular/core';
import { NgControl } from '@angular/forms';

@Directive({
  selector: '[appMoneyInput]'
})
export class MoneyInputDirective {
  @Input() allowEmpty: boolean = false;

  constructor(private el: ElementRef<HTMLInputElement>) {}

  ngAfterViewInit() {
    setTimeout(() => {
      this.formatValue(this.el.nativeElement.value);
      this.onBlur();
    });
  }

  @HostListener('input', ['$event.target.value'])
  onInput(value: string) {
    this.formatValue(value);
  }

  @HostListener('blur')
  onBlur() {
    let val = this.el.nativeElement.value.replace(/,/g, '').trim();
    if (!val) return;

    let [intPart, decimalPart = ''] = val.split('.');

    // จำกัดทศนิยมไม่เกิน 2 หลัก
    decimalPart = decimalPart.substring(0, 2);

    // เติม .00 ถ้าไม่มีทศนิยม
    if (decimalPart === '') decimalPart = '00';

    // ใส่ comma ให้จำนวนเต็ม
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, ',');

    this.el.nativeElement.value = `${intPart}.${decimalPart}`;
  }

  formatValue(value: string) {
    // เก็บจุดไว้ ถ้ามีอยู่แล้วในข้อความ
    const hasDot = value.includes('.');

    // ลบอักขระที่ไม่ใช่ตัวเลขหรือจุด
    let cleaned = value.replace(/[^0-9.]/g, '');

    // จำกัดให้มีจุดทศนิยมได้แค่ 1 จุด
    const firstDot = cleaned.indexOf('.');
    if (firstDot !== -1) {
      cleaned =
        cleaned.slice(0, firstDot + 1) +
        cleaned.slice(firstDot + 1).replace(/\./g, '');
    }

    // กรณีเริ่มด้วยจุด เช่น ".5" → "0.5"
    if (cleaned.startsWith('.')) cleaned = '0' + cleaned;

    // ลบศูนย์นำหน้า (ถ้าไม่ใช่ "0." และยาวเกิน 1)
    if (!cleaned.startsWith('0.') && cleaned.length > 1) {
      cleaned = cleaned.replace(/^0+/, '');
    }

    if (!this.allowEmpty && cleaned === '') cleaned = '0';

    let [intPart, decimalPart = ''] = cleaned.split('.');

    // จำกัดทศนิยมไม่เกิน 2 หลัก
    decimalPart = decimalPart.substring(0, 2);

    // ใส่ comma ให้จำนวนเต็ม
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, ',');

    // รวมกลับ โดยถ้าผู้ใช้เพิ่งพิม "." ให้คงไว้ (เพื่อให้พิมพ์ต่อได้)
    this.el.nativeElement.value =
      hasDot && !value.endsWith('.') && decimalPart === ''
        ? `${intPart}.`
        : decimalPart !== ''
        ? `${intPart}.${decimalPart}`
        : value.endsWith('.')
        ? `${intPart}.`
        : intPart;
  }
}






// import {
//   Directive,
//   ElementRef,
//   HostListener,
//   Input,
//   OnChanges,
//   Optional,
//   SimpleChanges
// } from '@angular/core';
// import { NgControl } from '@angular/forms';

// @Directive({
//   selector: '[appMoneyInput]'
// })
// export class MoneyInputDirective implements OnChanges {
//   @Input() min?: string | number;
//   @Input() max?: string | number;
//   @Input() allowEmpty: boolean = false;
//   @Input() defaultValue: number | string = 0;
//   @Input() clearZeroOnFocus: boolean = false;
//   private hasChanged = false
//   private minValue?: number;
//   private maxValue?: number;

//   constructor(private el: ElementRef, @Optional() private control: NgControl) { }
//   ngOnInit() {
//     // แสดง defaultValue ตั้งแต่เข้าหน้า
//     if (!this.allowEmpty) {
//       const defaultNum = this.parseToNumber(this.defaultValue) ?? 0;
//       const formatted = this.formatNumber(defaultNum);
//       this.el.nativeElement.value = formatted;
//       this.control?.control?.setValue(formatted);
//     }
//   }
//   ngOnChanges(changes: SimpleChanges) {
//     if ('min' in changes) {
//       this.minValue = this.parseToNumber(this.min);
//     }
//     if ('max' in changes) {
//       this.maxValue = this.parseToNumber(this.max);
//     }
//   }

//   @HostListener('keydown', ['$event'])
//   onKeyDown(event: KeyboardEvent) {
//     if (event.key === ',' || event.key === 'Comma') {
//       event.preventDefault();
//     }
//   }

//   @HostListener('input', ['$event.target'])
//   onInput(target: HTMLInputElement) {
//     this.hasChanged = true
//     let value = target.value;

//     const endsWithDot = value.endsWith('.') && (value.match(/\./g) || []).length === 1;

//     // เอาเฉพาะตัวเลขกับจุด
//     let cleaned = '';
//     let dotSeen = false;
//     for (const char of value) {
//       if (char >= '0' && char <= '9') {
//         cleaned += char;
//       } else if (char === '.' && !dotSeen) {
//         cleaned += char;
//         dotSeen = true;
//       }
//     }

//     if (cleaned.startsWith('.')) {
//       cleaned = '0' + cleaned;
//     }

//     const parts = cleaned.split('.');
//     let intPart = parts[0].replace(/^0+/, '') || '0';
//     let decimalPart = parts[1] || '';
//     decimalPart = decimalPart.substring(0, 2);
//     intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, ',');

//     let formatted = decimalPart ? `${intPart}.${decimalPart}` : intPart;

//     if (endsWithDot && decimalPart.length === 0) {
//       formatted += '.';
//     }

//     const numericValue = parseFloat(cleaned);
//     if (!isNaN(numericValue)) {
//       if (this.minValue !== undefined && numericValue < this.minValue) {
//         formatted = this.formatNumber(this.minValue);
//       }
//       if (this.maxValue !== undefined && numericValue > this.maxValue) {
//         formatted = this.formatNumber(this.maxValue);
//       }
//     }

//     // ลบจนว่างได้ ถ้า allowEmpty
//     if (cleaned === '' && this.allowEmpty) {
//       formatted = '';
//     }

//     this.el.nativeElement.value = formatted;
//     this.control?.control?.setValue(formatted);

//     setTimeout(() => {
//       const pos = this.el.nativeElement.value.length;
//       this.el.nativeElement.setSelectionRange(pos, pos);
//     }, 0);
//   }

//   @HostListener('focus')
//   onFocus() {
//     if (!this.clearZeroOnFocus) return;

//     const value = this.el.nativeElement.value.replace(/,/g, '');
//     if (value === '0' || value === '0.00') {
//       this.el.nativeElement.value = '';
//       this.control?.control?.setValue('');
//     }
//   }


//   @HostListener('blur')
//   onBlur() {
//     let value = this.el.nativeElement.value;
//     const raw = value.replace(/,/g, '');

//     // ถ้าช่องว่างหรือไม่ใช่ตัวเลข → ใช้ defaultValue
//     if (!raw || isNaN(parseFloat(raw))) {
//       if (!this.allowEmpty) {
//         const defaultNum = this.parseToNumber(this.defaultValue);
//         const formatted = this.formatNumber(defaultNum ?? 0);
//         this.el.nativeElement.value = formatted;
//         this.control?.control?.setValue(formatted);
//       }
//       this.hasChanged = false;
//       return;
//     }

//     if (!this.hasChanged) return;

//     const parts = raw.split('.');
//     let intPart = parts[0].replace(/^0+/, '') || '0';
//     let decimalPart = parts[1] || '';

//     if (decimalPart.length === 0) {
//       decimalPart = '00';
//     } else if (decimalPart.length === 1) {
//       decimalPart += '0';
//     } else {
//       decimalPart = decimalPart.substring(0, 2);
//     }

//     intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, ',');

//     const formatted = `${intPart}.${decimalPart}`;
//     this.el.nativeElement.value = formatted;
//     this.control?.control?.setValue(formatted);
//     this.hasChanged = false;
//   }


//   private parseToNumber(value: string | number | undefined): number | undefined {
//     if (value === undefined || value === null) return undefined;
//     if (typeof value === 'number') return value;
//     const clean = value.replace(/,/g, '');
//     const num = parseFloat(clean);
//     return isNaN(num) ? undefined : num;
//   }

//   private formatNumber(value: number): string {
//     const parts = value.toFixed(2).split('.');
//     const intPart = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
//     const decimalPart = parts[1];
//     return `${intPart}.${decimalPart}`;
//   }
// }
